/**
 * 词典适配器 - 整合新旧词典系统
 * 提供向后兼容的接口，同时使用新的词典管理系统
 */
class DictionaryAdapter {
    constructor() {
        this.newManager = null;
        this.legacyData = {};
        this.isLoaded = false;
        this.loadPromise = null;
        this.enabledLanguages = {
            zh: false,
            en: false,
            fr: false,
            ru: false,
            es: false,
            ja: false
        };
        // 新增：词典ID级别的启用状态
        this.enabledDictionaries = {};
        
        // 新增：词典缓存机制
        this.dictionaryCache = new Map(); // 缓存合并后的词典
        this.cacheVersion = 0; // 缓存版本号，用于失效检测
        this.cacheExpiry = 10 * 60 * 1000; // 10分钟过期时间
    }

    /**
     * 初始化适配器
     * @returns {Promise<boolean>} 初始化是否成功
     */
    async initialize() {
        if (this.loadPromise) {
            return await this.loadPromise;
        }
        
        this.loadPromise = this._doInitialize();
        return await this.loadPromise;
    }

    /**
     * 执行初始化
     * @private
     */
    async _doInitialize() {
        try {
            // 初始化新的词典管理器
            if (typeof DictionaryManager !== 'undefined') {
                this.newManager = new DictionaryManager();
                await this.newManager.initialize();
                
                // 加载所有预设词典
                await this._loadAllPresetDictionaries();
                
                this.isLoaded = true;
                console.log('DictionaryAdapter initialized successfully');
                return true;
            } else {
                throw new Error('DictionaryManager not available');
            }
        } catch (error) {
            console.error('Failed to initialize DictionaryAdapter:', error);
            // 降级到传统加载方式
            await this._loadLegacyDictionaries();
            return false;
        }
    }

    /**
     * 加载所有预设词典
     * @private
     */
    async _loadAllPresetDictionaries() {
        try {
            // 加载预设词典
            const presetDictionaries = this.newManager.getAvailableDictionaries('preset', false);
            
            for (const dictInfo of presetDictionaries) {
                const dictData = await this.newManager.loadDictionary(dictInfo.id);
                if (dictData && dictData.words) {
                    // 转换为旧格式：{word: "pos"}
                    const converted = this._convertDictionaryFormat(dictData);
                    this.legacyData[dictInfo.language] = converted;
                    
                    console.log(`Loaded dictionary: ${dictInfo.language} (${Object.keys(converted).length} words)`);
                }
            }
            
            // 加载本地词典
            const localDictionaries = this.newManager.getAvailableDictionaries('local', false);
            
            for (const dictInfo of localDictionaries) {
                const dictData = await this.newManager.loadDictionary(dictInfo.id);
                if (dictData && dictData.words) {
                    // 转换为旧格式：{word: "pos"}
                    const converted = this._convertDictionaryFormat(dictData);
                    
                    // 合并到对应语言的词典中
                    if (!this.legacyData[dictInfo.language]) {
                        this.legacyData[dictInfo.language] = {};
                    }
                    Object.assign(this.legacyData[dictInfo.language], converted);
                    
                    console.log(`Loaded local dictionary: ${dictInfo.displayName} (${Object.keys(converted).length} words)`);
                }
            }
            
            // 加载下载的词典
            const downloadedDictionaries = this.newManager.getAvailableDictionaries('downloaded', false);
            
            for (const dictInfo of downloadedDictionaries) {
                const dictData = await this.newManager.loadDictionary(dictInfo.id);
                if (dictData && dictData.words) {
                    // 转换为旧格式：{word: "pos"}
                    const converted = this._convertDictionaryFormat(dictData);
                    
                    // 合并到对应语言的词典中
                    if (!this.legacyData[dictInfo.language]) {
                        this.legacyData[dictInfo.language] = {};
                    }
                    Object.assign(this.legacyData[dictInfo.language], converted);
                    
                    console.log(`Loaded downloaded dictionary: ${dictInfo.displayName} (${Object.keys(converted).length} words)`);
                }
            }
        } catch (error) {
            console.error('Error loading dictionaries:', error);
            throw error;
        }
    }

    /**
     * 转换词典格式：从新格式转换为旧格式
     * @param {Object} dictData 新格式词典数据
     * @returns {Object} 旧格式词典数据
     * @private
     */
    _convertDictionaryFormat(dictData) {
        const converted = {};
        if (dictData.words) {
            for (const [word, info] of Object.entries(dictData.words)) {
                if (info.pos && info.pos.length > 0) {
                    converted[word] = info.pos[0]; // 取第一个词性
                }
            }
        }
        return converted;
    }

    /**
     * 降级到传统词典加载方式
     * @private
     */
    async _loadLegacyDictionaries() {
        try {
            console.log('Falling back to legacy dictionary loading...');
            
            const languages = ['EN', 'ZH', 'FR', 'ES', 'RU', 'JA'];
            const loadPromises = languages.map(async (lang) => {
                try {
                    const response = await fetch(chrome.runtime.getURL(`dictionaries/${lang}_word.json`));
                    const data = await response.json();
                    const langCode = lang.toLowerCase();
                    this.legacyData[langCode] = this._convertDictionaryFormat(data);
                    return { 
                        lang: langCode, 
                        count: Object.keys(this.legacyData[langCode]).length, 
                        version: data.version || '1.0'
                    };
                } catch (error) {
                    console.warn(`Failed to load ${lang} dictionary:`, error);
                    return { lang: lang.toLowerCase(), count: 0, version: 'error' };
                }
            });
            
            const results = await Promise.all(loadPromises);
            
            this.isLoaded = true;
            console.log('Legacy dictionaries loaded:', results.reduce((acc, result) => {
                acc[`${result.lang}_words`] = result.count;
                return acc;
            }, {}));
            
        } catch (error) {
            console.error('Legacy dictionary loading failed:', error);
            this._loadFallbackDictionaries();
        }
    }

    /**
     * 加载备用词典（简化版）
     * @private
     */
    _loadFallbackDictionaries() {
        console.log('Using fallback dictionaries');
        
        this.legacyData.en = {
            'computer': 'n', 'book': 'n', 'table': 'n', 'person': 'n',
            'good': 'a', 'bad': 'a', 'big': 'a', 'small': 'a',
            'run': 'v', 'jump': 'v', 'read': 'v', 'write': 'v'
        };
        
        this.legacyData.zh = {
            '电脑': 'n', '书': 'n', '桌子': 'n', '人': 'n',
            '好': 'a', '坏': 'a', '大': 'a', '小': 'a',
            '跑': 'v', '跳': 'v', '读': 'v', '写': 'v'
        };
        
        this.isLoaded = true;
    }

    // ========== 向后兼容的接口 ==========

    /**
     * 获取指定语言的词典
     * @param {string} language 语言代码
     * @returns {Object} 词典数据
     */
    getDictionary(language) {
        const cacheKey = `${language}_${this.cacheVersion}`;
        const cachedEntry = this.dictionaryCache.get(cacheKey);
        if (cachedEntry) {
            const now = Date.now();
            if (now - cachedEntry.timestamp < this.cacheExpiry) {
                return cachedEntry.data;
            } else {
                this.dictionaryCache.delete(cacheKey);
            }
        }

        let result = {};
        const selectedIds = Object.entries(this.enabledDictionaries)
            .filter(([, v]) => !!v)
            .map(([k]) => k);

        if (this.newManager && selectedIds.length > 0) {
            const enabledDicts = this.newManager
                .getDictionariesByLanguage(language, false)
                .filter(dict => selectedIds.includes(dict.id));
            if (enabledDicts.length > 0) {
                enabledDicts.sort((a, b) => (a.priority || 999) - (b.priority || 999));
                const mergedDict = {};
                for (const dict of enabledDicts) {
                    const dictData = this.newManager.getDictionaryData(dict.id);
                    if (dictData) {
                        Object.assign(mergedDict, dictData);
                    }
                }
                result = mergedDict;
            }
        }

        this.dictionaryCache.set(cacheKey, {
            data: result,
            timestamp: Date.now()
        });
        return result;
    }

    /**
     * 检查词典是否已加载
     * @returns {boolean} 是否已加载
     */
    isReady() {
        return this.isLoaded;
    }

    /**
     * 等待词典加载完成
     * @returns {Promise<boolean>} 加载结果
     */
    async waitForLoad() {
        if (this.isLoaded) return true;
        if (this.loadPromise) return await this.loadPromise;
        return await this.initialize();
    }

    /**
     * 查找词汇的词性
     * @param {string} word 要查找的词汇
     * @param {string} language 语言代码
     * @returns {string|null} 词性或null
     */
    lookupWord(word, language) {
        const dictionary = this.getDictionary(language);
        return dictionary[word] || null;
    }

    /**
     * 更新启用的语言列表（旧接口，向后兼容）
     * @param {Object} enabledLanguages 启用的语言设置
     */
    updateEnabledLanguages(enabledLanguages) {
        console.log('Updating enabled languages (legacy):', enabledLanguages);
        
        // 检查是否为新格式（包含词典ID）
        const hasNewFormat = Object.keys(enabledLanguages).some(key => key.includes('-'));
        
        if (hasNewFormat) {
            // 新格式：使用词典ID
            this.updateEnabledDictionaries(enabledLanguages);
        } else {
            // 旧格式：使用语言代码
            this.enabledLanguages = { ...this.enabledLanguages, ...enabledLanguages };
            
            // 如果使用新管理器，同步更新注册表中的启用状态
            if (this.newManager) {
                this._syncEnabledLanguagesToRegistry(enabledLanguages);
            }
        }
    }

    /**
     * 更新启用的词典列表（新接口）
     * @param {Object} enabledDictionaries 启用的词典设置
     */
    updateEnabledDictionaries(enabledDictionaries) {
        console.log('Updating enabled dictionaries:', enabledDictionaries);
        this.enabledDictionaries = { ...enabledDictionaries };
        
        // 清除缓存并更新版本号
        this.invalidateCache();
        
        // 同步更新语言状态
        this._updateLanguageStatusFromDictionaries();
    }

    // 注意：updateEnabledLanguages 已在上方定义（处理新/旧格式）。此处去重，防止被覆盖。

    /**
     * 清除词典缓存
     */
    invalidateCache() {
        // 清理所有缓存
        this.dictionaryCache.clear();
        this.cacheVersion++;
        console.log(`Dictionary cache invalidated, new version: ${this.cacheVersion}`);
    }

    /**
     * 清理过期的缓存条目
     * @private
     */
    _cleanExpiredCache() {
        const now = Date.now();
        for (const [key, entry] of this.dictionaryCache.entries()) {
            if (now - entry.timestamp >= this.cacheExpiry) {
                this.dictionaryCache.delete(key);
            }
        }
    }

    /**
     * 同步启用语言设置到注册表
     * @param {Object} enabledLanguages 启用的语言设置
     * @private
     */
    _syncEnabledLanguagesToRegistry(enabledLanguages) {
        // TODO: 实现注册表更新逻辑
        // 这里可以调用新管理器的updateRegistry方法
        console.log('Syncing enabled languages to registry:', enabledLanguages);
    }

    /**
     * 根据启用的词典更新语言状态
     * @private
     */
    _updateLanguageStatusFromDictionaries() {
        if (!this.newManager) return;
        
        // 重置语言状态
        const languageStatus = {
            zh: false,
            en: false,
            fr: false,
            ru: false,
            es: false,
            ja: false
        };
        
        console.log('Updating language status from dictionaries:', this.enabledDictionaries);
        
        // 检查每个语言是否有启用的词典
        Object.keys(this.enabledDictionaries).forEach(dictId => {
            if (this.enabledDictionaries[dictId]) {
                const dict = this.newManager.getDictionaryById(dictId);
                console.log(`Dictionary ${dictId}:`, dict);
                
                // 特别调试111词典 - 检查两个可能的ID
                if (dictId === 'custom-1760195631107' || dictId === 'custom-1760202653658') {
                    console.log('🔍 Found 111 dictionary in enabled list!');
                    console.log('Dictionary object:', dict);
                    console.log('Dictionary language:', dict ? dict.language : 'undefined');
                }
                
                if (dict && dict.language) {
                    languageStatus[dict.language] = true;
                    console.log(`✅ Enabled language: ${dict.language} (from dictionary: ${dictId})`);
                } else {
                    console.log(`❌ Dictionary ${dictId} not found or has no language property`);
                }
            }
        });
        
        // 更新语言启用状态
        this.enabledLanguages = languageStatus;
        
        console.log('Updated language status from dictionaries:', this.enabledLanguages);
    }

    /**
     * 检查语言是否启用
     * @param {string} language 语言代码
     * @returns {boolean} 是否启用
     */
    isLanguageEnabled(language) {
        return this.enabledLanguages[language] || false;
    }

    /**
     * 获取启用的语言列表
     * @returns {Array<string>} 启用的语言代码数组
     */
    getEnabledLanguages() {
        return Object.keys(this.enabledLanguages).filter(lang => this.enabledLanguages[lang]);
    }

    /**
     * 获取所有已加载词典的统计信息
     * @returns {Object} 统计信息
     */
    getStatistics() {
        const stats = {
            totalLanguages: Object.keys(this.legacyData).length,
            totalWords: 0,
            languages: {},
            usingNewManager: !!this.newManager
        };

        for (const [lang, dict] of Object.entries(this.legacyData)) {
            const wordCount = Object.keys(dict).length;
            stats.languages[lang] = wordCount;
            stats.totalWords += wordCount;
        }

        return stats;
    }

    // ========== 新功能接口 ==========

    /**
     * 获取词典元数据
     * @param {string} language 语言代码
     * @returns {Object|null} 词典元数据
     */
    getDictionaryMeta(language) {
        if (!this.newManager) return null;
        
        const dictionaries = this.newManager.getDictionariesByLanguage(language);
        return dictionaries.length > 0 ? dictionaries[0] : null;
    }

    /**
     * 获取所有可用词典信息
     * @returns {Array} 词典信息列表
     */
  getAvailableDictionaries() {
        if (!this.newManager) {
            // 返回传统格式的词典信息
            return Object.keys(this.legacyData).map(lang => ({
                language: lang,
                name: lang.toUpperCase(),
                enabled: this.enabledLanguages[lang] || false,
                wordCount: Object.keys(this.legacyData[lang]).length
            }));
    }

    return this.newManager.getAvailableDictionaries('all', false);
  }

  getEnabledDictionaryIds(language) {
    if (!this.newManager) return [];
    const ids = this.newManager
      .getDictionariesByLanguage(language, false)
      .filter(d => !!this.enabledDictionaries[d.id])
      .map(d => d.id)
      .sort();
    return ids;
  }

    /**
     * 重新加载词典
     * @param {string} language 语言代码，不提供则重新加载所有
     * @returns {Promise<boolean>} 重新加载是否成功
     */
    async reload(language = null) {
        // 清除缓存
        this.invalidateCache();
        
        if (language) {
            // 重新加载特定语言词典
            if (this.newManager) {
                const dictionaries = this.newManager.getDictionariesByLanguage(language);
                if (dictionaries.length > 0) {
                    this.newManager.clearCache(dictionaries[0].id);
                    const dictData = await this.newManager.loadDictionary(dictionaries[0].id, false);
                    if (dictData) {
                        this.legacyData[language] = this._convertDictionaryFormat(dictData);
                        return true;
                    }
                }
            }
            return false;
        } else {
            // 重新加载所有词典
            this.isLoaded = false;
            this.loadPromise = null;
            if (this.newManager) {
                this.newManager.clearCache();
            }
            return await this.initialize();
        }
    }
}

// 导出适配器
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { DictionaryAdapter };
} else if (typeof window !== 'undefined') {
    window.DictionaryAdapter = DictionaryAdapter;
}